unit fROR_XMLReport;

interface

uses
  Windows, Messages, SysUtils, Variants, Classes, Graphics, Controls, Forms,
  Dialogs, fROR_MDIChild, OleCtrls, SHDocVw, EmbeddedWB, ActnList,
  Menus, VA508AccessibilityManager;

type
  TFormXMLReport = class(TFormMDIChild)
    mnuPopup: TPopupMenu;
    miBack: TMenuItem;
    miForward: TMenuItem;
    miCancel: TMenuItem;
    N1: TMenuItem;
    miCopy: TMenuItem;
    miSelectAll: TMenuItem;
    N2: TMenuItem;
    miTextSize: TMenuItem;
    miLargest: TMenuItem;
    miLarger: TMenuItem;
    miMedium: TMenuItem;
    miSmall: TMenuItem;
    miSmallest: TMenuItem;
    N3: TMenuItem;
    miFind: TMenuItem;
    dlgSaveDialog: TSaveDialog;
    ActionList: TActionList;
    acCopy: TAction;
    acSelectAll: TAction;
    acBack: TAction;
    acForward: TAction;
    acFind: TAction;
    acCancel: TAction;
    EmbeddedWB: TEmbeddedWB;
    VA508AccessibilityManager1: TVA508AccessibilityManager;
    procedure acBackExecute(Sender: TObject);
    procedure acCancelExecute(Sender: TObject);
    procedure acCopyExecute(Sender: TObject);
    procedure acFindExecute(Sender: TObject);
    procedure acForwardExecute(Sender: TObject);
    procedure acSelectAllExecute(Sender: TObject);
    procedure dlgSaveDialogTypeChange(Sender: TObject);
    procedure EmbeddedWBBeforeNavigate2(aSender: TObject; const pDisp: IDispatch;
      var URL, Flags, TargetFrameName, PostData, Headers: OleVariant;
      var Cancel: WordBool);
    procedure EmbeddedWBCommandStateChange(ASender: TObject;
      Command: Integer; Enable: WordBool);
    procedure EmbeddedWBNewWindow2(ASender: TObject; var ppDisp: IDispatch;
      var Cancel: WordBool);
    function EmbeddedWBShowContextMenu(const dwID: Cardinal;
      const ppt: PPoint; const pcmdtReserved: IInterface;
      const pdispReserved: IDispatch): HRESULT;
    procedure EmbeddedWBStatusTextChange(ASender: TObject;
      const Text: WideString);
    procedure EmbeddedWBTitleChange(ASender: TObject;
      const Text: WideString);
    function EmbeddedWBTranslateUrl(const dwTranslate: Cardinal;
      const pchURLIn: PWideChar; var ppchURLOut: PWideChar): HRESULT;
    function EmbeddedWBUpdateUI: HRESULT;
    procedure FormClose(Sender: TObject; var Action: TCloseAction);
    procedure miZoomClick(Sender: TObject);

  private
    SortMode: TStringList;
    Task: String;

    function  GetReportURL(): String;

  public
    constructor Create(anOwner: TComponent); override;
    constructor CreateReport(aTask: String; anOwner: TComponent = nil);
    destructor  Destroy; override;

    function  CanBePrinted: Boolean; override;
    function  HasBeenModified: Boolean; override;
    procedure LoadReport;
    function  PageSetup: Boolean; override;
    function  Print: Boolean; override;
    function  PrintPreview: Boolean; override;
    function  SaveAs: Boolean; override;
    function  SaveAsCSV(Output: TStream): Boolean;
    function  SaveAsHTML(Output: TStream): Boolean;
    function  SaveAsXML(Output: TStream): Boolean;

    class procedure View(aTask: String; aSortMode: TStringList = nil);

  published

    property ReportURL: String read GetReportURL;

  end;

implementation
{$R *.dfm}

uses
  uROR_Utilities, uROR_Common, uROR_Broker, fROR_XMLEngine,
  uROR_NamespaceHandler, fROR_Main, uROR_MultiStream;

var
  NumRptWindows: Integer = 0;
  PreviousZoom: Integer = -999999;

constructor TFormXMLReport.Create(anOwner: TComponent);
begin
  inherited;
  SortMode := TStringList.Create;
  Task := '';
end;

constructor TFormXMLReport.CreateReport(aTask: String; anOwner: TComponent);
begin
  if Assigned(anOwner) then
    Create(anOwner)
  else
    Create(Application);

  Task := aTask;
  Inc(NumRptWindows);
end;

destructor TFormXMLReport.Destroy;
begin
  FreeAndNil(SortMode);
  Dec(NumRptWindows);
  if (NumRptWindows = 0) and (PreviousZoom <> -999999) then
    begin
      try
        EmbeddedWB.Zoom(PreviousZoom);
      except
      end;
      PreviousZoom := -999999;
    end;
  inherited Destroy;
end;

procedure TFormXMLReport.acBackExecute(Sender: TObject);
begin
  try EmbeddedWB.GoBack; except end;
end;

procedure TFormXMLReport.acCancelExecute(Sender: TObject);
begin
  try EmbeddedWB.Stop; except end;
end;

procedure TFormXMLReport.acCopyExecute(Sender: TObject);
begin
  try EmbeddedWB.Copy; except end;
end;

procedure TFormXMLReport.acFindExecute(Sender: TObject);
begin
  try EmbeddedWB.Find; except end;
end;

procedure TFormXMLReport.acForwardExecute(Sender: TObject);
begin
  try EmbeddedWB.GoForward; except end;
end;

procedure TFormXMLReport.acSelectAllExecute(Sender: TObject);
begin
  try EmbeddedWB.SelectAll; except end;
end;

function TFormXMLReport.CanBePrinted: Boolean;
begin
  Result := True;
end;

procedure TFormXMLReport.dlgSaveDialogTypeChange(Sender: TObject);
begin
  with dlgSaveDialog do
    case FilterIndex of
      2:   DefaultExt := 'htm';
      3:   DefaultExt := 'xml';
      else DefaultExt := 'csv';
    end;
end;

procedure TFormXMLReport.EmbeddedWBBeforeNavigate2(aSender: TObject;
  const pDisp: IDispatch; var URL, Flags, TargetFrameName, PostData,
  Headers: OleVariant; var Cancel: WordBool);
begin
  if Pos(sReportNameSpace, URL) = 1 then
    DisplayStatus('Loading and transforming the report...');
end;

procedure TFormXMLReport.EmbeddedWBCommandStateChange(aSender: TObject;
  Command: Integer; Enable: WordBool);
begin
  case Command of
    CSC_NAVIGATEBACK: acBack.Enabled := Enable;
    CSC_NAVIGATEFORWARD: acForward.Enabled := Enable;
    else acCancel.Enabled := EmbeddedWB.Busy;
  end;
end;

procedure TFormXMLReport.EmbeddedWBNewWindow2(aSender: TObject;
  var ppDisp: IDispatch; var Cancel: WordBool);
var
  form: TFormXMLReport;
begin
  // Open the link in new report window
  form := TFormXMLReport.CreateReport(Task);
  ppDisp := form.EmbeddedWB.Application;
end;

function TFormXMLReport.EmbeddedWBShowContextMenu(const dwID: Cardinal;
  const ppt: PPoint; const pcmdtReserved: IUnknown;
  const pdispReserved: IDispatch): HRESULT;
var
  i, n, zmax, zoom: Integer;
begin
  n := miTextSize.Count - 1;

  if Not miTextSize.Enabled then
    begin
      zoom := EmbeddedWB.ZoomRangeLow;
      zmax := EmbeddedWB.ZoomRangeHigh;
      for i:=n downto 0  do
        begin
          miTextSize.Items[i].Tag := zoom;
          if zoom > zmax then miTextSize.Items[i].Enabled := False;
          Inc(zoom);
        end;
      miTextSize.Enabled := True;
    end;

  zoom := EmbeddedWB.ZoomValue;
  n := miTextSize.Count - 1;
  for i:= 0 to n do
    if miTextSize.Items[i].Tag = zoom then
       miTextSize.Items[i].Checked := True;

  mnuPopup.Popup(ppt.x, ppt.y);
  Result := S_OK;
end;

procedure TFormXMLReport.EmbeddedWBStatusTextChange(aSender: TObject;
  const Text: WideString);
begin
  DisplayStatus(Text);
end;

procedure TFormXMLReport.EmbeddedWBTitleChange(aSender: TObject;
  const Text: WideString);
begin
  Caption := Text;
end;

function TFormXMLReport.EmbeddedWBTranslateUrl(const dwTranslate: Cardinal;
  const pchURLIn: PWideChar; var ppchURLOut: PWideChar): HRESULT;
var
  buf, mode: String;
begin
  if ParseReportURL(pchURLIn, mode, buf, SortMode) then
    begin
      ppchURLOut := StringToOleStr(ReportURL);
      Result := S_OK;
    end
  else
    begin
      ppchURLOut := nil;
      Result := S_FALSE;
    end;
end;

function TFormXMLReport.EmbeddedWBUpdateUI: HRESULT;
begin
  acCopy.Enabled := EmbeddedWB.OleObject.Document.queryCommandEnabled('Copy');
  Result := S_OK;
end;

procedure TFormXMLReport.FormClose(Sender: TObject; var Action: TCloseAction);
begin
  Action := caFree;
end;

function TFormXMLReport.GetReportURL(): String;
var
  params: String;
  i, n: Integer;
begin
  params := '';
  n := SortMode.Count - 1;
  for i:=0 to n do params := params + '&' + SortMode.Strings[i];
  if Pos('&', params) = 1 then params[1] := '?';
  Result := sReportNameSpace + 'task/' + Task + params;
end;

function TFormXMLReport.HasBeenModified: Boolean;
begin
  Result := True;
end;

procedure TFormXMLReport.LoadReport;
begin
  DisplayStatus('Loading embeded browser...');
  EmbeddedWB.Go(ReportURL);
  if PreviousZoom = -999999 then
    try
      PreviousZoom := EmbeddedWB.ZoomValue;
    except
    end;
  EmbeddedWB.SetFocusToDOc;
end;

procedure TFormXMLReport.miZoomClick(Sender: TObject);
begin
  with (Sender as TMenuitem) do
    begin
      EmbeddedWB.Zoom(Tag);
      Checked := True;
    end;
end;

function TFormXMLReport.PageSetup: Boolean;
begin
  try
    EmbeddedWB.PageSetup(False);
    Result := True;
  except
    Result := False;
  end;
end;

function TFormXMLReport.Print: Boolean;
var
  vaIn, vaOut: OleVariant;
begin
  try
    EmbeddedWB.ControlInterface.ExecWB(OLECMDID_PRINT,
          OLECMDEXECOPT_PROMPTUSER, vaIn, vaOut);
    Result := True;
  except
    Result := False;
  end;
end;

function TFormXMLReport.PrintPreview: Boolean;
begin
  try
    EmbeddedWB.PrintPreview;
    Result := True;
  except
    Result := False;
  end;
end;

function TFormXMLReport.SaveAs: Boolean;
var
  output: TStream;
  prevcursor: TCursor;
begin
  Result := False;
  if dlgSaveDialog.Execute then
    begin
      Application.ProcessMessages;
      prevcursor := Screen.Cursor;
      Screen.Cursor := crHourglass;
      try
        if dlgSaveDialog.FilterIndex = 1 then
          output := TMultiFileStream.Create(dlgSaveDialog.FileName)
        else
          output := TFileStream.Create(dlgSaveDialog.FileName, fmCreate);
        case dlgSaveDialog.FilterIndex of
          2:   Result := SaveAsHTML(output);
          3:   Result := SaveAsXML(output);
          else Result := SaveAsCSV(output);
        end;
        FreeAndNil(output);
      finally
        Screen.Cursor := prevcursor;
      end;
    end;
end;

function TFormXMLReport.SaveAsCSV(Output: TStream): Boolean;
begin
  with TXMLEngine.Create(Broker) do
    begin
      try
        Result := TransformReport(Task, Output, '2', SortMode);
      except
        Result := False
      end;
      Free;
    end;
end;

function TFormXMLReport.SaveAsHTML(Output: TStream): Boolean;
begin
{  EmbeddedWB.SaveToStream(Output);
  Result := True;}
  with TXMLEngine.Create(Broker) do
    begin
      SablotEngine.SetParam('REMOVE_URLS', '1');
      try
        Result := TransformReport(Task, Output, '1', SortMode);
      except
        Result := False;
      end;
      Free;
    end;
end;

function TFormXMLReport.SaveAsXML(Output: TStream): Boolean;
var
  from: String;
  rpcbuf: TStringList;
begin
  rpcbuf := TStringList.Create;
  try
    from := '';
    repeat
      Result := Broker.CallProc(rpcReportRetrieve,
         [Task,from,'16384B'], SortMode, rpcbuf);
      if Not Result then
        Exit;
      Application.ProcessMessages;
      from := rpcbuf.Strings[0];
      rpcbuf.Delete(0);
      rpcbuf.SaveToStream(Output);
    until Piece(from,'^') = '';
  finally
    rpcbuf.Free;
  end;
end;

class procedure TFormXMLReport.View(aTask: String; aSortMode: TStringList);
begin
  with TFormXMLReport.CreateReport(aTask) do
    begin
      if Assigned(aSortMode) then
        SortMode.Assign(aSortMode);
      LoadReport;
    end;
end;

end.
